//
//  HSBWheelHelpers.swift
//  Do It
//
//  Created by Jim Dovey on 2/10/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

protocol ColorInfo: Identifiable {
    static var predefined: [Self] { get }
    var hsb: (Double, Double, Double) { get set }
    var uiColor: SwiftUI.Color { get }
    var localizedName: LocalizedStringKey { get }
}

extension ColorInfo {
    var hue: Double {
        get {
            let (h, _, _) = hsb
            return h
        }
        set {
            let (_, s, b) = hsb
            hsb = (newValue, s, b)
        }
    }

    var saturation: Double {
        get {
            let (_, s, _) = hsb
            return s
        }
        set {
            let (h, _, b) = hsb
            hsb = (h, newValue, b)
        }
    }

    var brightness: Double {
        get {
            let (_, _, b) = hsb
            return b
        }
        set {
            let (h, s, _) = hsb
            hsb = (h, s, newValue)
        }
    }
}

extension TodoItemList.Color: ColorInfo {
    var hsb: (Double, Double, Double) {
        get {
            let uicolor: UIColor
            switch self {
            case let .custom(hue, saturation, brightness):
                return (hue, saturation, brightness)

            case .red:
                uicolor = UIColor.systemRed
            case .green:
                uicolor = UIColor.systemGreen
            case .blue:
                uicolor = UIColor.systemBlue
            case .orange:
                uicolor = UIColor.systemOrange
            case .yellow:
                uicolor = UIColor.systemYellow
            case .pink:
                uicolor = UIColor.systemPink
            case .purple:
                uicolor = UIColor.systemPurple
            }

            var (h, s, b) = (CGFloat(0), CGFloat(0), CGFloat(0))
            uicolor.getHue(&h, saturation: &s, brightness: &b, alpha: nil)
            return (Double(h), Double(s), Double(b))
        }
        set {
            let (h, s, b) = newValue
            self = .custom(hue: h, saturation: s, brightness: b)
        }
    }

    static var predefined: [TodoItemList.Color] = [
        .red, .green, .blue, .orange, .yellow, .pink, .purple
    ]

    var id: Self { self }
    
    var localizedName: LocalizedStringKey {
        switch self {
        case .red: return "red"
        case .green: return "green"
        case .blue: return "blue"
        case .orange: return "orange"
        case .yellow: return "yellow"
        case .pink: return "pink"
        case .purple: return "purple"
        case .custom: return "custom color"
        }
    }
}

enum HSB {
    static func hueSaturation(at unitLocation: CGPoint) -> (Double, Double) {
        let (radius, angle) = unitLocation.distanceAndAngle()
        let saturation = Double(radius)
        var hue = Double(angle)
        if unitLocation.y < 0 {
            hue = 1.0 - hue
        }
        return (hue, saturation)
    }

    static func components<C: ColorInfo>(at unitLocation: CGPoint, basedOn color: C) -> (Double, Double, Double) {
        let (hue, saturation) = hueSaturation(at: unitLocation)
        return (hue, saturation, color.brightness)
    }

    static func updateColor<C: ColorInfo>(_ color: inout C, at unitLocation: CGPoint) {
        var (h, s, b) = color.hsb
        (h, s) = hueSaturation(at: unitLocation)
        color.hsb = (h, s, b)
    }

    static func uiColor<C: ColorInfo>(at unitLocation: CGPoint, basedOn color: C) -> SwiftUI.Color {
        let (h, s, b) = components(at: unitLocation, basedOn: color)
        return Color(hue: h, saturation: s, brightness: b)
    }

    static func offset<C: ColorInfo>(for color: C) -> CGSize {
        let (hue, saturation, _) = color.hsb

        // Unit distance from center to the radius appropriate
        // for our saturation.
        let radius = CGFloat(saturation)

        // angle appropriate to the chosen hue.
        let angle = CGFloat(hue * .pi * 2)

        return CGPoint.zero.offset(by: radius, at: angle)
    }

    static func unitOffset<C: ColorInfo>(for color: C, within size: CGSize) -> CGSize {
        let dimension = min(size.width, size.height)
        let selectionOffset = offset(for: color)
        return CGSize(width: (dimension/2) * selectionOffset.width,
                      height: (dimension/2) * selectionOffset.height)
    }
    
    static func unitPosition<C: ColorInfo>(for color: C, within size: CGSize) -> CGPoint {
        let offset = unitOffset(for: color, within: size)
        let rect = CGRect(origin: .zero, size: size)
        let center = CGPoint(x: rect.midX, y: rect.midY)
        return center + offset
    }
}
